#ifndef _RTC_DRIVER_EXT_H_
#define _RTC_DRIVER_EXT_H_

#include "rtc_driver_int.h"

#define RTC_CLKOUT_DISABLE  0
#define RTC_CLKOUT_1HZ     1
#define RTC_CLKOUT_OSC_CLOCK  2

#define RTC_DAYLIGHTSAVINGS_DISABLE  0
#define RTC_DAYLIGHTSAVINGS_ENABLE  1

#define RTC_ALARM_MATCH_SMH 0    //Alarm matches only seconds, minutes & hours
#define RTC_ALARM_MATCH_SMHD 1   //Alarm matches only seconds, minutes, hours & days
#define RTC_ALARM_MATCH_SMHDM 2  //Alarm matches only seconds, minutes, hours, days & months
#define RTC_ALARM_MATCH_SMHDMY 3 //Alarm matches only seconds, minutes, hours, days, months & years

#define RTC_ISR_FRE512HZ  IRTC_ISR_H_SAM7_MASK
#define RTC_ISR_FRE256HZ  IRTC_ISR_H_SAM6_MASK
#define RTC_ISR_FRE128HZ  IRTC_ISR_H_SAM5_MASK
#define RTC_ISR_FRE64HZ   IRTC_ISR_H_SAM4_MASK
#define RTC_ISR_FRE32HZ   IRTC_ISR_H_SAM3_MASK
#define RTC_ISR_FRE16HZ   IRTC_ISR_H_SAM2_MASK
#define RTC_ISR_FRE8HZ    IRTC_ISR_H_SAM1_MASK
#define RTC_ISR_FRE4HZ    IRTC_ISR_H_SAM0_MASK
#define RTC_ISR_FRE2HZ    IRTC_ISR_L_2HZ_MASK
#define RTC_ISR_FRE1HZ    IRTC_ISR_L_1HZ_MASK
#define RTC_ISR_MIN       IRTC_ISR_L_MIN_MASK
#define RTC_ISR_HR        IRTC_ISR_L_HR_MASK
#define RTC_ISR_DAY       IRTC_ISR_L_DAY_MASK
#define RTC_ISR_ALM       IRTC_ISR_L_ALM_MASK
#define RTC_ISR_CDT       IRTC_ISR_L_CDT_MASK

#define RTC_ISR_BAT_TMPR  0x1
#define RTC_ISR_TMPR1     0x4
#define RTC_ISR_TMPR2     0x2

enum month_names {
  January = 1,
  February,
  March,
  April,
  May,
  June,
  July,
  August,
  September,
  October,
  November,
  December
};

enum weekday_names {
  Sunday = 0,
  Monday,
  Tuesday,
  Wednesday,
  Thursday,
  Friday,
  Saturday
};

/******************************************************************************/
/* FUNCTION          : rtc_setConfig                                          */
/******************************************************************************/
/* Abstract          : This function configs the control register of IRTC     */
/*                                                                            */
/* Input Parameters  : rtc_clkout: RTC clock out                              */
/*                     daylightsavings_enable: DST enabled/disabled           */
/*                     softreset: if 1 then RTC is soft reset                 */
/*                     callback: function ptr for ISR callback                */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setConfig(unsigned char rtc_clkout, unsigned char daylightsavings_enable
                            , unsigned char softreset, void (*func)(unsigned int, unsigned char));

/******************************************************************************/
/* FUNCTION          : rtc_setAlarmConfig                                     */
/******************************************************************************/
/* Abstract          : This function configs the alarm control register       */
/*                                                                            */
/* Input Parameters  : alarm_match: alarm configuration be used               */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setAlarmConfig(unsigned char alarm_match);

/******************************************************************************/
/* FUNCTION          : rtc_setCompensation                                    */
/******************************************************************************/
/* Abstract          : This function sets the compensation settings for RTC   */
/*                                                                            */
/* Input Parameters  : comp_interval: compensation interval window            */
/*                     comp_value: 2's complement form of compensating value  */
/*                                                                            */
/* Return Parameter  : none                                                   */
/******************************************************************************/
void rtc_setCompensation(unsigned char comp_interval, unsigned char comp_value);

/******************************************************************************/
/* FUNCTION          : rtc_setYear                                            */
/******************************************************************************/
/* Abstract          : This function sets the year in IRTC                    */
/*                                                                            */
/* Input Parameters  : year: year to be set                                   */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setYear(unsigned int year);

/******************************************************************************/
/* FUNCTION          : rtc_setMonth                                           */
/******************************************************************************/
/* Abstract          : This function sets the month in IRTC                   */
/*                                                                            */
/* Input Parameters  : month: month to be set                                 */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setMonth(enum month_names month);

/******************************************************************************/
/* FUNCTION          : rtc_setDayAndDate                                      */
/******************************************************************************/
/* Abstract          : This function sets the day and weekday in IRTC         */
/*                                                                            */
/* Input Parameters  : weekday: week day to be set                            */
/*                     date: date to be set                                   */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setDayAndDate(enum weekday_names weekday, unsigned char date);

/******************************************************************************/
/* FUNCTION          : rtc_setHour                                            */
/******************************************************************************/
/* Abstract          : This function sets the hour in IRTC in 24H time format */
/*                                                                            */
/* Input Parameters  : hour: hour to be set                                   */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setHour(unsigned char hour);

/******************************************************************************/
/* FUNCTION          : rtc_setMin                                             */
/******************************************************************************/
/* Abstract          : This function sets the minutes in IRTC                 */
/*                                                                            */
/* Input Parameters  : min: minutes to be set                                 */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setMin(unsigned char min);

/******************************************************************************/
/* FUNCTION          : rtc_setSeconds                                         */
/******************************************************************************/
/* Abstract          : This function sets the seconds in IRTC                 */
/*                                                                            */
/* Input Parameters  : sec: seconds to be set                                 */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setSeconds(unsigned char sec);

/******************************************************************************/
/* FUNCTION          : rtc_setDateAndTime                                     */
/******************************************************************************/
/* Abstract          : This is one function that sets the date and time both  */
/*                                                                            */
/* Input Parameters  : year: year to be programmed                            */
/*                     month: month to be programmed                          */
/*                     date:  date  to be programmed                          */
/*                     weekday: day to be programmed                          */
/*                     hour : hour  to be programmed                          */
/*                     minutes: minutes to be programmed                      */
/*                     seconds: seconds to be programmed                      */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setDateAndTime(unsigned int year, enum month_names month, unsigned char date,
                                 enum weekday_names weekday, unsigned char hour, unsigned char minutes,
                                 unsigned char seconds);

/******************************************************************************/
/* FUNCTION          : rtc_getYear                                            */
/******************************************************************************/
/* Abstract          : This function gets the year in IRTC                    */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : year value                                             */
/******************************************************************************/
unsigned int rtc_getYear(void);

/******************************************************************************/
/* FUNCTION          : rtc_getMonth                                           */
/******************************************************************************/
/* Abstract          : This function gets the month in IRTC                   */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : month value                                            */
/******************************************************************************/
enum month_names rtc_getMonth(void);

/******************************************************************************/
/* FUNCTION          : rtc_getDayAndDate                                      */
/******************************************************************************/
/* Abstract          : This function gets the day and weekday in IRTC         */
/*                                                                            */
/* Output Parameters  : weekday: week day to be get                           */
/*                     date: date to be get                                   */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
void rtc_getDayAndDate(enum weekday_names *weekday, unsigned char * date);

/******************************************************************************/
/* FUNCTION          : rtc_getHour                                            */
/******************************************************************************/
/* Abstract          : This function gets the hour in IRTC in 24H time format */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : hour value                                             */
/******************************************************************************/
unsigned char rtc_getHour(void);

/******************************************************************************/
/* FUNCTION          : rtc_getMin                                             */
/******************************************************************************/
/* Abstract          : This function gets the minutes in IRTC                 */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : minutes value                                          */
/******************************************************************************/
unsigned char rtc_getMin(void);

/******************************************************************************/
/* FUNCTION          : rtc_getSeconds                                         */
/******************************************************************************/
/* Abstract          : This function gets the seconds in IRTC                 */
/*                                                                            */
/* Input Parameters  : sec: seconds to be get                                 */
/*                                                                            */
/* Return Parameter  : seconds value                                          */
/******************************************************************************/
unsigned char rtc_getSeconds(void);

/******************************************************************************/
/* FUNCTION          : rtc_setAlarm                                           */
/******************************************************************************/
/* Abstract          : This is one function that sets alarm in IRTC           */
/*                                                                            */
/* Input Parameters  : year: year to be programmed                            */
/*                     month: month to be programmed                          */
/*                     date:  date  to be programmed                          */
/*                     hour : hour  to be programmed                          */
/*                     minutes: minutes to be programmed                      */
/*                     seconds: seconds to be programmed                      */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setAlarm(unsigned int year, enum month_names month, unsigned char date,
                           unsigned char hour, unsigned char minutes, unsigned char seconds);

/******************************************************************************/
/* FUNCTION          : rtc_getAlarmYear                                       */
/******************************************************************************/
/* Abstract          : This function gets alarm year programmed in IRTC       */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : year value                                             */
/******************************************************************************/
unsigned int rtc_getAlarmYear(void);

/******************************************************************************/
/* FUNCTION          : rtc_getAlarmMonth                                      */
/******************************************************************************/
/* Abstract          : This function gets alarm month programmed in IRTC      */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : month value                                            */
/******************************************************************************/
#define rtc_getAlarmMonth()  irtc_getAlarmMonth()

/******************************************************************************/
/* FUNCTION          : rtc_getDayAndDate                                      */
/******************************************************************************/
/* Abstract          : This function gets alarm date programmed in RTC        */
/*                                                                            */
/* Output Parameters : none                                                   */
/*                                                                            */
/* Return Parameter  : date value                                             */
/******************************************************************************/
#define rtc_getAlarmDate()   irtc_getAlarmDate()

/******************************************************************************/
/* FUNCTION          :  rtc_getAlarmHour                                      */
/******************************************************************************/
/* Abstract          : This function gets alarm hour in IRTC in 24H format    */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : hour value                                             */
/******************************************************************************/
#define rtc_getAlarmHour()   irtc_getAlarmHour()

/******************************************************************************/
/* FUNCTION          : rtc_getAlarmMin                                        */
/******************************************************************************/
/* Abstract          : This function gets alarm minutes in IRTC               */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : minutes value                                          */
/******************************************************************************/
#define rtc_getAlarmMin() irtc_getAlarmMin()

/******************************************************************************/
/* FUNCTION          : rtc_getAlarmSeconds                                    */
/******************************************************************************/
/* Abstract          : This function gets alarm seconds in IRTC               */
/*                                                                            */
/* Input Parameters  : sec: seconds to be get                                 */
/*                                                                            */
/* Return Parameter  : seconds value                                          */
/******************************************************************************/
#define rtc_getAlarmSeconds()   irtc_getAlarmSeconds()

/******************************************************************************/
/* FUNCTION          : rtc_setBatteryTamperConfig                             */
/******************************************************************************/
/* Abstract          : This function configs battery tamper control           */
/*                                                                            */
/* Input Parameters  : state: whether ON or OFF                               */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setBatteryTamperConfig(unsigned char state);

/******************************************************************************/
/* FUNCTION          : rtc_setTamper1Config                                   */
/******************************************************************************/
/* Abstract          : This function configs the Tamper1 control and filter   */
/*                                                                            */
/* Input Parameters  : state: whether ON or OFF                               */
/*                     filter_clk: clock select for filter delay              */
/*                     filter_duration: duration for filter delay             */
/*                     pol: polarity, 1-active low, 0-active high             */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setTamper1Config(unsigned char state, unsigned char filter_clk
                                  , unsigned char filter_duration, unsigned char pol);

/******************************************************************************/
/* FUNCTION          : rtc_setTamper2Config                                   */
/******************************************************************************/
/* Abstract          : This function configs the Tamper2 control and filter   */
/*                                                                            */
/* Input Parameters  : state: whether ON or OFF                               */
/*                     filter_clk: clock select for filter delay              */
/*                     filter_duration: duration for filter delay             */
/*                     pol: polarity, 1-active low, 0-active high             */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setTamper2Config(unsigned char state, unsigned char filter_clk
                                  , unsigned char filter_duration, unsigned char pol);

/******************************************************************************/
/* FUNCTION          : rtc_getTamperStampYear                                 */
/******************************************************************************/
/* Abstract          : This function gets tamper stamp year in IRTC           */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : year value                                             */
/******************************************************************************/
unsigned int rtc_getTamperStampYear(void);

/******************************************************************************/
/* FUNCTION          : rtc_getTamperStampMonth                                */
/******************************************************************************/
/* Abstract          : This function gets tamper stamp month in IRTC          */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : month value                                            */
/******************************************************************************/
#define rtc_getTamperStampMonth()  irtc_getTamperStampMonth()

/******************************************************************************/
/* FUNCTION          : rtc_getDayAndDate                                      */
/******************************************************************************/
/* Abstract          : This function gets tamper stamp date in RTC            */
/*                                                                            */
/* Output Parameters : none                                                   */
/*                                                                            */
/* Return Parameter  : date value                                             */
/******************************************************************************/
#define rtc_getTamperStampDate()   irtc_getTamperStampDate()

/******************************************************************************/
/* FUNCTION          :  rtc_getTamperStampHour                                */
/******************************************************************************/
/* Abstract          : This function gets tamper stamp hour in 24H format     */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : hour value                                             */
/******************************************************************************/
#define rtc_getTamperStampHour()   irtc_getTamperStampHour()

/******************************************************************************/
/* FUNCTION          : rtc_getTamperStampMin                                  */
/******************************************************************************/
/* Abstract          : This function gets tamper stamp minutes in IRTC        */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : minutes value                                          */
/******************************************************************************/
#define rtc_getTamperStampMin() irtc_getTamperStampMin()

/******************************************************************************/
/* FUNCTION          : rtc_getTamperStampSeconds                              */
/******************************************************************************/
/* Abstract          : This function gets tamper stamp  seconds in IRTC       */
/*                                                                            */
/* Input Parameters  : sec: seconds to be get                                 */
/*                                                                            */
/* Return Parameter  : seconds value                                          */
/******************************************************************************/
#define rtc_getTamperStampSeconds()   irtc_getTamperStampSeconds()

/******************************************************************************/
/* FUNCTION          : rtc_setCountDownTimer                                  */
/******************************************************************************/
/* Abstract          : This function sets the seconds in IRTC                 */
/*                                                                            */
/* Input Parameters  : minutes: no. of minutes to be countdown                */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setCountDownTimer(unsigned char minutes);

/******************************************************************************/
/* FUNCTION          : rtc_EnableInterrupt                                    */
/******************************************************************************/
/* Abstract          : This function sets particular interrupt in IRTC        */
/*                                                                            */
/* Input Parameters  : interrupt_byte: interrupt mask to be enabled           */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_EnableInterrupt(unsigned int interrupt_byte);

/******************************************************************************/
/* FUNCTION          : rtc_DisableInterrupt                                   */
/******************************************************************************/
/* Abstract          : This function disables particular interrupt in IRTC    */
/*                                                                            */
/* Input Parameters  : interrupt_byte: interrupt mask to be enabled           */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_DisableInterrupt(unsigned int interrupt_byte);

/******************************************************************************/
/* FUNCTION          : rtc_clearInterrupt                                     */
/******************************************************************************/
/* Abstract          : This function clears particular interrupt in IRTC      */
/*                                                                            */
/* Input Parameters  : interrupt_byte: interrupt mask to be cleared           */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_clearInterrupt(unsigned int interrupt_byte);

#endif  /* #ifndef _RTC_DRIVER_EXT_H_ */
